/**
 * \file: exchnd_ctl.c
 * \file: exchnd_test.c
 *
 * Exception handler tests
 *
 * \component: exchnd
 *
 * \author: Matthias Weise (mweise@de.adit-jv.com)
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <getopt.h>

#include <linux/exchnd.h>

#include <sys/prctl.h>

#include "exchnd_interface.h"
#include "exchnd_names.h"

#define DEFAULT_DEV_NAME "/dev/exchnd"

#define EXCHND_TYPE "exchnd_ctl"
#ifdef EXCHND_TEST
#   include <sys/types.h>
#   include <signal.h>
#   undef EXCHND_TYPE
#   define EXCHND_TYPE "exchnd_test"
#   define TESTS_USAGE                                              \
    "usage: "EXCHND_TYPE " [-d <device>] --fork\n"                  \
    "usage: "EXCHND_TYPE " [-d <device>] --threads\n"               \
    "usage: "EXCHND_TYPE " [-d <device>] --oom\n"                   \
    "usage: "EXCHND_TYPE " [-d <device>] --segv\n"                  \
    "usage: "EXCHND_TYPE " [-d <device>] --ill\n"                   \
    "usage: "EXCHND_TYPE " [-d <device>] --div0\n"                  \
    "usage: "EXCHND_TYPE " [-d <device>] --overflow\n"              \
    "usage: "EXCHND_TYPE " [-d <device>] --abort\n"                 \
    "usage: "EXCHND_TYPE " [-d <device>] --recurse\n"               \
    "usage: "EXCHND_TYPE " [-d <device>] --recovery\n"              \
    "usage: "EXCHND_TYPE " [-d <device>] --raise\n"                 \
    "usage: "EXCHND_TYPE " [-d <device>] --on-demand\n"
#   define TESTS_DESC                                               \
    "    --fork           fork bomb\n"                              \
    "    --threads        fork once and sleep\n"                    \
    "    --oom            trigger an oom exception\n"               \
    "    --segv           trigger a segmentation fault by Memory\n" \
    "    --segv2          trigger a segmentation fault by PC\n"     \
    "    --ill            trigger an illegal instruction fault\n"   \
    "    --div0           trigger a division by zero\n"             \
    "    --overflow       trigger a buffer overflow\n"              \
    "    --abort          trigger abort.\n"                         \
    "    --recurse        trigger recursive function call.\n"       \
    "    --recovery       trigger read pointer recovery.\n"         \
    "    --raise          Raise signal using raise().\n"            \
    "    --on-demand      trigger on demand event.\n"
#else
#   define TESTS_USAGE
#   define TESTS_DESC
#endif

#define USAGE                                                                  \
    "Default device name is: " DEFAULT_DEV_NAME ". \n"                         \
    "usage: "EXCHND_TYPE " [-d <device>] -g <trigger>\n"                       \
    "usage: "EXCHND_TYPE " [-d <device>] -c <trigger> <module>...\n"           \
    "usage: "EXCHND_TYPE " [-d <device>] --get-sig SIGSEGV\n"                  \
    "usage: "EXCHND_TYPE \
    " [-d <device>] --set-sig SIGSEGV BACKTRACE STACK_DUMP\n"                  \
    "usage: "EXCHND_TYPE " [-d <device>] --get-def-config[=EC_DEFAULT_1]\n"    \
    "usage: "EXCHND_TYPE " [-d <device>] --set-def-config=EC_DEFAULT_1\n"      \
    "usage: "EXCHND_TYPE " [-d <device>] --get-sig-config[=ESC_DEFAULT_1]\n"   \
    "usage: "EXCHND_TYPE " [-d <device>] --set-sig-config=ESC_DEFAULT_1\n"     \
    "usage: "EXCHND_TYPE " [-d <device>] --get-sig-mask\n"                     \
    "usage: "EXCHND_TYPE " [-d <device>] --set-sig-mask 0xffffffef\n"          \
    "usage: "EXCHND_TYPE " [-d <device>] --set-app-config <module>"            \
    " <library path>\n"   \
    "usage: "EXCHND_TYPE " [-d <device>] --configure-syscalls=enable\n"        \
    "usage: "EXCHND_TYPE " [-d <device>] --configure-taskswitches=enable\n"    \
    "usage: "EXCHND_TYPE " [-d <device>] --configure-hist-size=128\n"          \
    "usage: "EXCHND_TYPE " [-d <device>] --configure-hist-pid=234\n"           \
    "usage: "EXCHND_TYPE " [-d <device>] --list-triggers\n"                    \
    "usage: "EXCHND_TYPE " [-d <device>] --list-modules\n"                     \
    "usage: "EXCHND_TYPE " [-d <device>] --list-signals\n"                     \
    TESTS_USAGE                                                                \
    "\n"                                                                       \
                                                                               \
    "    -d, --device <device>\n"                                              \
    "             select exception handler device\n"                           \
    "    -g, --get-config <trigger> [ -g ... ]\n"                              \
    "             get the configuration for a trigger\n"                       \
    "    -c, --set-config trigger> <module> ..."                               \
    " [-c <trigger> <module> ...]\n"                                           \
    "             set the configuration for a trigger\n"                       \
    "    -G[<ID>], --get-def-config[=<ID>]\n"                                  \
    "             get the trigger's configuration set\n"                       \
    "    -C, --set-def-config <ID>\n"                                          \
    "             set the default triggers' configuration set\n\n"             \
                                                                               \
                                                                               \
    "    --get-sig <ID> [--get-sig <ID> ...]\n"                                \
    "              get the configuration of a signal\n"                        \
    "    --set-sig <ID> <modules> [--set-sig <ID> <modules> ...]\n"            \
    "              set the configuration for a signal\n"                       \
    "    -s, --get-sig-config[=<ID>]\n"                                        \
    "             get default signals configuration set\n"                     \
    "    -S, --set-sig-config <ID>\n"                                          \
    "             set default signals configuration set\n\n"                   \
                                                                               \
                                                                               \
    "    --get-sig-mask \n"                                                    \
    "              get the mask for signals\n"                                 \
    "    --set-sig-mask <mask>\n"                                              \
    "              set the mask for a signals.\n"                              \
    "              Interpreted as hex value.\n\n"                              \
                                                                               \
    "    --set-app-config <module> <library path>\n"                           \
    "              Sets path to the library for the specified application \n"  \
    "              specific module.\n"                                         \
    "    e.g.: --set-app-config APPLICATION_SPECIFIC1 /home/root/lib.so\n"     \
    "              If complete path is not given, the daemon will first\n"     \
    "              search in /lib and then in /usr/lib. \n\n"                  \
                                                                               \
                                                                               \
    "     -a, --add <process name>\n"                                          \
    "             add a kernel process to filter white list\n"                 \
    "     -r, --remove <process name>\n"                                       \
    "             remove a kernel process from filter white list\n"            \
    "     -A, --add-path <path to binary>\n"                                   \
    "             add a user process to filter white list\n"                   \
    "     -R, --remove-path <path to binary>\n"                                \
    "             remove a user process from filter white list\n\n"            \
                                                                               \
    "    Collector modules related actions:\n"                                 \
    "    --configure-syscalls=[disable/enable]\n"                              \
    "    --configure-taskswitches=[disable/enable]\n"                          \
    "       disable\n"                                                         \
    "             Disable related collector (default).\n"                      \
    "       enable\n"                                                          \
    "             Enable collector. May slow down the system.\n"               \
    "    --configure-hist-size=[size]\n"                                       \
    "             Configure the amount of data to provide in logs.\n"          \
    "             64 elements by default. Bounded to 1024 elements.\n"         \
    "             Affects both EHM_HIST_SYSCALLS and EHM_HIST_TASKSWITCHES\n"  \
    "    --configure-hist-pid=[PID/0]\n"                                       \
    "             Configure a PID to watch. The PID given is considered as\n"  \
    "             the group leader (main process). If 0 is provided all\n"     \
    "             tasks of the system are watched.\n"                          \
    "             Affects EHM_HIST_SYSCALLS and EHM_HIST_TASKSWITCHES\n\n"     \
                                                                               \
    "    Listing:\n"                                                           \
    "    --list-triggers  list supported triggers\n"                           \
    "    --list-modules   list supported modules\n"                            \
    "    --list-signals   list supported signals\n\n"                          \
                                                                               \
    TESTS_DESC

#ifdef EXCHND_TEST
#   define THREAD_SLEEP_SEC 100
#endif

/* usage - command line help */
static void usage()
{
    printf(USAGE);
}

enum exchnd_action {
    EXCHND_HELP = 1,
#ifdef EXCHND_TEST
    EXCHND_FORK,
    EXCHND_THREAD,
    EXCHND_OOM,
    EXCHND_SEGV,
    EXCHND_SEGV2,
    EXCHND_ILLEGAL,
    EXCHND_DIV_ZERO,
    EXCHND_OVERFLOW,
    EXCHND_ABORT,
    EXCHND_RECURSE,
    EXCHND_RECOVERY,
    EXCHND_RAISE,
    EXCHND_ON_DEMAND,
#endif
    EXCHND_FILTER,
    EXCHND_GET_CONF,
    EXCHND_SET_CONF,
    EXCHND_GET_SIG,
    EXCHND_SET_SIG,
    EXCHND_GET_SIG_MASK,
    EXCHND_SET_SIG_MASK,
    EXCHND_GET_DEF_CONF,
    EXCHND_SET_DEF_CONF,
    EXCHND_GET_SIG_CONF,
    EXCHND_SET_SIG_CONF,
    EXCHND_LIST_TRIGGERS,
    EXCHND_LIST_MODULES,
    EXCHND_LIST_SIGNALS,
    EXCHND_SET_APP_SPEC,
    EXCHND_SYSCALLS,
    EXCHND_TASKSWITCHES,
    EXCHND_HIST_SIZE,
    EXCHND_HIST_PID,
    EXCHND_NO_ACTION
};

static int action = EXCHND_HELP;

static char *dev_name = DEFAULT_DEV_NAME;

#define MAX_NUM_CONF 50
static int last_conf = 0;

static struct exchnd_conf_param configs[MAX_NUM_CONF];
static struct exchnd_conf_filter filter[MAX_NUM_CONF];

static char *library_path = NULL;
enum exchnd_modules app_spec_module = EHM_NONE;

#ifdef EXCHND_TEST

/* catch_signal - signal handler */
static void catch_signal(int signo)
{
    switch (signo) {
    case SIGINT:
    case SIGQUIT:
    case SIGTERM:
    case SIGSEGV:
        printf("Signal %d received.\n", signo);
        break;
    }

    signal(signo, SIG_DFL);
    kill(getpid(), signo);
}

/* install_signal_handler - install a handler for some signals */
static void install_signal_handler()
{
    int signals[] = { SIGINT, SIGQUIT, SIGTERM, SIGSEGV, SIGUSR2, 0 };
    unsigned int i;
    struct sigaction sa;

    /* install a signal handler for the above listed signals */
    for (i = 0; signals[i]; i++) {
        memset(&sa, 0, sizeof(sa));
        sa.sa_handler = catch_signal;

        if (sigaction(signals[i], &sa, NULL) < 0)
            printf("Failed to install signal %u handler. Error: %u\n",
                   i, signals[i]);
    }
}

static void *sleep_thread(void *arg)
{
    int *num = arg;
    char name[16] = "\0";

    snprintf(name, sizeof(name), "Thread num. %.2i", *num);
    prctl(PR_SET_NAME, name, NULL, NULL, NULL);

    /* Let's try to generate an error. */
    usleep(THREAD_SLEEP_SEC * 1000000);
    return arg;
}

static void function_abortlast(const int myvalue,
                               int prev1,
                               int prev2,
                               int prev3,
                               char *tp)
{
    printf(tp, myvalue, prev1, prev2, prev3);
    fflush(stdout);
    abort();
}

static void function_abort3(const int myvalue, int prev1, int prev2)
{
    const int ANOTHER_CONST = myvalue + 10;
    char template[] = "99%% done ... And then ... %d (%d) (%d) (%d)?\n";
    printf("80%% done ... Will it reach %d ?\n", myvalue);
    fflush(stdout);
    function_abortlast(ANOTHER_CONST, prev1 + 2, prev2 + 3, myvalue + 4,
                       template);
}

static void function_abort2(const int myvalue, int prev1)
{
    const int ANOTHER_CONST = myvalue + 60;
    printf("25%% done ... Will it outstrip %d ?\n", myvalue);
    fflush(stdout);
    function_abort3(ANOTHER_CONST, prev1 + 2, myvalue + 3);
}

static void function_abort1(const int myvalue)
{
    const int ANOTHER_CONST = myvalue + 15;
    printf("10%% done ... Will it outstrip %d ?\n", myvalue);
    fflush(stdout);
    function_abort2(ANOTHER_CONST, myvalue + 2);
}

static void function_recurse()
{
    static int count = 200;

    if (count--)
        function_recurse();
    else
        usleep(THREAD_SLEEP_SEC * 1000000);
}

static int sig = SIGSEGV; /* For Raise signal Test */
static void exchnd_raise_signal(int sig)
{
    pid_t pid;
    printf("Inside the main, create process by fork()\n");
    pid = fork();

    if (0 == pid) {
        printf("In Child process\n");
        raise(sig);
    } else {
        printf("Child Process %d created\n", pid);
        sleep(1);
    }
}

int pid;
static void exchnd_on_demand(int pid)
{
#   if EXCHND_VERSION > 0x211
#      include <sys/ioctl.h>
    struct exchnd_on_demand param;
    int exh_fd = exchnd_create_device(dev_name, O_RDONLY);
    int rc;

    if (exh_fd <= 0) {
        printf("Unable to open device: %s\n", strerror(errno));
        return;
    }

    if (pid) {
        param.pid = pid;
        param.modules[0] = EHM_NONE;
        snprintf(param.msg, 256, "Getting backtrace of process %d:", pid);
    } else {
        param.pid = getpid();
        param.use_def = 0;
        param.modules[0] = EHM_PROCESSOR_REGISTERS;
        param.modules[1] = EHM_STACK_DUMP;
        param.modules[2] = EHM_THREAD_LIST;
        param.modules[3] = EHM_BACKTRACE_ALL_THREADS;
        param.modules[4] = EHM_CPU_USAGE;
        param.modules[5] = EHM_NONE;
        strcpy(param.msg, "I'm an EXCHND_ON_DEMAND test.");
    }

    rc = ioctl(exh_fd, IOCTL_EXCHND_ON_DEMAND, &param);

    if (rc < 0)
        printf("Unable to call on demand: %s\n", strerror(errno));

#   else
    pid = pid; /* To suppress unused warning */
    printf("On demand not implemented.\n");
#   endif
}
#endif

int filter_config(char **argv)
{
    char *modules_arg = NULL;
    enum exchnd_modules (*conf)[EHM_LAST_ELEMENT + 1] = NULL;
    unsigned int n = 1, n_max = 1;
    enum exchnd_modules i = EHM_NONE;
    enum exchnd_modules conf_filter[EHM_LAST_ELEMENT];

    conf = &configs[last_conf].modules;

    memset(conf_filter, 0, (EHM_LAST_ELEMENT)*sizeof(int));

    /* Selecting valid arguments */
    while (argv[optind] != NULL &&
           argv[optind][0] != '-') {
        enum exchnd_modules temp_mod = EHM_NONE;
        modules_arg = argv[optind++];
        /* Being user friendly. */
        exchnd_normalize(modules_arg);
        temp_mod = get_module(modules_arg);

        if (temp_mod == EHM_LAST_ELEMENT) {
            printf("Ignoring unknown module %s\n", modules_arg);
            continue;
        }

        if (conf_filter[temp_mod]) {
            printf("Ignoring duplicate %s\n", modules_arg);
            continue;
        }

        conf_filter[temp_mod] = (enum exchnd_modules)n_max++;
    }

    /* Recover user ordering */
    while (n != n_max)
        for (i = EHM_NONE; i < EHM_LAST_ELEMENT; i++)
            if (conf_filter[i] == n)
                (*conf)[n++] = i;

    return n;
}

int parse_arguments(int argc, char **argv)
{
    int c;

#define EXCHND_SHORT_ARGS "hd:g:c:G::C:s::S:a:r:A:R:"
    static struct option long_options[] = {
        /* Set flag options. */
#ifdef EXCHND_TEST
        { "fork", no_argument, &action, EXCHND_FORK },
        { "threads", no_argument, &action, EXCHND_THREAD },
        { "oom", no_argument, &action, EXCHND_OOM },
        { "segv", no_argument, &action, EXCHND_SEGV },
        { "segv2", no_argument, &action, EXCHND_SEGV2 },
        { "ill", no_argument, &action, EXCHND_ILLEGAL },
        { "div0", no_argument, &action, EXCHND_DIV_ZERO },
        { "overflow", no_argument, &action, EXCHND_OVERFLOW },
        { "abort", no_argument, &action, EXCHND_ABORT },
        { "recurse", no_argument, &action, EXCHND_RECURSE },
        { "recovery", no_argument, &action, EXCHND_RECOVERY },
        { "raise", optional_argument, &action, EXCHND_RAISE },
        { "on-demand", optional_argument, &action, EXCHND_ON_DEMAND },
#endif
        { "list-triggers", no_argument, &action, EXCHND_LIST_TRIGGERS },
        { "list-modules", no_argument, &action, EXCHND_LIST_MODULES },
        { "list-signals", no_argument, &action, EXCHND_LIST_SIGNALS },

        /* Need processing options. */
        { "help", no_argument, NULL, 'h' },
        { "get-config", required_argument, NULL, 'g' },
        { "set-config", required_argument, NULL, 'c' },
        { "get-sig", required_argument, &action, EXCHND_GET_SIG },
        { "set-sig", required_argument, &action, EXCHND_SET_SIG },
        { "get-def-config", optional_argument, NULL, 'G' },
        { "set-def-config", required_argument, NULL, 'C' },
        { "get-sig-config", optional_argument, NULL, 's' },
        { "set-sig-config", required_argument, NULL, 'S' },
        { "get-sig-mask", no_argument, &action, EXCHND_GET_SIG_MASK },
        { "set-sig-mask", required_argument, &action, EXCHND_SET_SIG_MASK },
        { "set-app-config", required_argument, &action, EXCHND_SET_APP_SPEC },
        { "configure-syscalls", required_argument, &action, EXCHND_SYSCALLS },
        { "configure-taskswitches",
          required_argument, &action, EXCHND_TASKSWITCHES },
        { "configure-hist-size", required_argument, &action, EXCHND_HIST_SIZE },
        { "configure-hist-pid", required_argument, &action, EXCHND_HIST_PID },
        { "device", required_argument, NULL, 'd' },
        { "add", required_argument, NULL, 'a' },
        { "remove", required_argument, NULL, 'r' },
        { "add-path", required_argument, NULL, 'A' },
        { "remove-path", required_argument, NULL, 'R' },

        /* List Terminator. He'll be back. */
        { 0, 0, 0, 0 }
    };

    c = getopt_long(argc, argv, EXCHND_SHORT_ARGS, long_options, NULL);

    while (c >= 0) {
        switch (c) {
        /* Set flag options. */
        case 0:

            /* Need processing options. */
            switch (action) {
            case EXCHND_GET_SIG:
            {
                int sig;

                if (last_conf >= MAX_NUM_CONF)
                    break;

                /* Being user friendly. */
                exchnd_normalize(optarg);

                sig = get_signal(optarg);

                if (sig > SIGSYS) {
                    printf("Unknown signal: %s\n", optarg);
                    return -EINVAL;
                }

                configs[last_conf].conf_type =
                    EXCHND_CONF_SIGNAL;
                configs[last_conf++].modules[0] = (enum exchnd_modules)sig;
                break;
            }
            case EXCHND_SET_SIG:
            {
                unsigned int n = 0;
                int sig = SIGSYS + 1;

                if (last_conf >= MAX_NUM_CONF)
                    break;

                /* Being user friendly. */
                exchnd_normalize(optarg);

                sig = get_signal(optarg);

                if (sig > SIGSYS) {
                    printf("Unknown signal: %s\n", optarg);
                    return -EINVAL;
                }

                configs[last_conf].conf_type =
                    EXCHND_CONF_SIGNAL;

                configs[last_conf].modules[0] = (enum exchnd_modules)sig;

                if (argv[optind] == NULL)
                    printf("Removing all modules for"
                           " signal %s (%d).\n",
                           optarg,
                           sig);

                /* Filtering arguments.
                 * Select each module needed only once.
                 * Ignoring unknown modules names.
                 */
                n = filter_config(argv);

                configs[last_conf++].modules[n] = EHM_NONE;
                break;
            }
            case EXCHND_GET_SIG_MASK:
            {
                if (last_conf >= MAX_NUM_CONF)
                    break;

                configs[last_conf++].conf_type = EXCHND_CONF_SIGHDL_MASK;
                break;
            }

            case EXCHND_SET_SIG_MASK:
            {
                int mask;

                if (last_conf >= MAX_NUM_CONF)
                    break;

                mask = strtoul(optarg, NULL, 16);

                configs[last_conf].conf_type = EXCHND_CONF_SIGHDL_MASK;
                configs[last_conf++].modules[0] = (enum exchnd_modules)mask;
                break;
            }
            case EXCHND_SET_APP_SPEC:
            {
                enum exchnd_modules module = EHM_NONE;
                char *module_arg = NULL;

                if (!argv[optind])
                    break;

                module_arg = optarg;
                exchnd_normalize(module_arg);
                module = get_module(module_arg);

                if ((module < EHM_APPLICATION_SPECIFIC1) ||
                    (module > EHM_APPLICATION_SPECIFIC5)) {
                    printf("Please provide one of the application specific"
                           " modules.\n");
                    break;
                }

                app_spec_module = module;
                library_path = argv[optind];
                break;
            }
            case EXCHND_TASKSWITCHES:
            case EXCHND_SYSCALLS:
            {
                if (last_conf >= MAX_NUM_CONF)
                    break;

#ifdef IOCTL_EXCHND_MODULES

                if (strncmp(optarg, "enable", 6) == 0)
                    configs[last_conf].conf_type = EXCHND_CONF_MODULE_ENABLE;
                else
                    configs[last_conf].conf_type = EXCHND_CONF_MODULE_DISABLE;

#endif

                if (action == EXCHND_SYSCALLS)
                    configs[last_conf++].modules[0] = EHM_HIST_SYSCALLS;
                else
                    configs[last_conf++].modules[0] = EHM_HIST_TASKSWITCHES;

                break;
            }
            case EXCHND_HIST_SIZE:
            {
#ifdef IOCTL_EXCHND_MODULES
                configs[last_conf].conf_type = EXCHND_CONF_MODULE_HIST_SIZE;
                configs[last_conf++].size = atoi(optarg);
#endif

                break;
            }
            case EXCHND_HIST_PID:
            {
#ifdef IOCTL_EXCHND_MODULES
                configs[last_conf].conf_type = EXCHND_CONF_MODULE_HIST_PID;
                configs[last_conf++].pid = atoi(optarg);
#endif

                break;
            }
#ifdef EXCHND_TEST
            case EXCHND_RAISE:
            {
                if (optarg != NULL) {
                    /* Being user friendly. */
                    exchnd_normalize(optarg);
                    sig = get_signal(optarg);

                    if (sig > SIGSYS) {
                        printf("Unknown signal: %s\n", optarg);
                        return -EINVAL;
                    }
                }

                break;
            }
            case EXCHND_ON_DEMAND:
            {
                if (optarg != NULL)
                    pid = atoi(optarg);
                else
                    pid = 0;

                break;
            }
#endif
            default:
                break;
            }

            break;
        case 'g':
        {
            enum exchnd_triggers trigger;
            action = EXCHND_GET_CONF;
            /* Being user friendly. */
            exchnd_normalize(optarg);

            trigger = get_trigger(optarg);

            if (trigger >= ET_LAST_ELEMENT) {
                printf("Unknown trigger: %s\n", optarg);
                return -EINVAL;
            }

            configs[last_conf].conf_type = EXCHND_CONF_TRIGGER;
            configs[last_conf++].modules[0] = (enum exchnd_modules)trigger;

            break;
        }
        case 'c':
        {
            unsigned int n = 0;
            enum exchnd_triggers trigger = ET_LAST_ELEMENT;

            if (last_conf >= MAX_NUM_CONF)
                break;

            action = EXCHND_SET_CONF;

            /* Being user friendly. */
            exchnd_normalize(optarg);

            trigger = get_trigger(optarg);

            if (trigger >= ET_LAST_ELEMENT) {
                printf("Unknown trigger: %s\n", optarg);
                return -EINVAL;
            }

            configs[last_conf].modules[0] = (enum exchnd_modules)trigger;

            if (argv[optind] == NULL)
                printf("Removing all modules for"
                       " trigger %s (%d).\n",
                       optarg,
                       trigger);

            /* Filtering arguments.
             * Select each module needed only once.
             * Ignoring unknown modules names.
             */
            n = filter_config(argv);

            configs[last_conf++].modules[n] = EHM_NONE;
            break;
        }
        case 'd':
            dev_name = optarg;
            printf("Trying to use device %s.\n", dev_name);
            break;
        case 'a':
        case 'r':
        case 'A':
        case 'R':
        {
            optind--;

            while (argv[optind] != NULL &&
                   argv[optind][0] != '-') {

                if (last_conf >= MAX_NUM_CONF)
                    break;

                strncpy(filter[last_conf].id,
                        argv[optind++],
                        MAX_FILTER_LENGTH);
                filter[last_conf].is_kernel = 0;

                switch (c) {
                case 'a':
                    filter[last_conf].is_kernel = 1;
                    filter[last_conf++].add = 1;
                    break;
                case 'A':
                    filter[last_conf++].add = 1;
                    break;
                case 'r':
                    filter[last_conf].is_kernel = 1;
                    filter[last_conf++].add = 0;
                    break;
                case 'R':
                    filter[last_conf++].add = 0;
                    break;
                default:
                    filter[last_conf++].add = 0;
                }
            }

            action = EXCHND_FILTER;
            break;
        }

        case 's':
        {
            int val = 0;

            if (optarg != NULL) {
                exchnd_normalize(optarg);
                val = get_sig_conf(optarg);
            }

            configs[last_conf].modules[0] = (enum exchnd_modules)val;
            configs[last_conf++].conf_type = EXCHND_CONF_SIGNAL_SET;
            action = EXCHND_GET_SIG_CONF;
            break;
        }
        case 'S':
            exchnd_normalize(optarg);
            configs[last_conf].modules[0] =
                (enum exchnd_modules)get_sig_conf(optarg);
            configs[last_conf++].conf_type = EXCHND_CONF_SIGNAL_SET;
            action = EXCHND_SET_SIG_CONF;
            break;

        case 'G':
        {
            int val = 0;

            if (optarg != NULL) {
                exchnd_normalize(optarg);
                val = get_conf(optarg);
            }

            configs[last_conf].conf_type = EXCHND_CONF_TRIGGER_SET;
            configs[last_conf++].modules[0] = (enum exchnd_modules)val;
            action = EXCHND_GET_DEF_CONF;
            break;
        }
        case 'C':
            exchnd_normalize(optarg);
            configs[last_conf].conf_type = EXCHND_CONF_TRIGGER_SET;
            configs[last_conf++].modules[0] =
                (enum exchnd_modules)get_conf(optarg);
            action = EXCHND_SET_DEF_CONF;
            break;
        /* Help/Unknown options. */
        case '?':
        case 'h':
        default:
            action = EXCHND_HELP;
        }

        c = getopt_long(argc, argv, EXCHND_SHORT_ARGS, long_options, NULL);
    }

    return 0;
}

/* -- Main */
int main(int argc, char **argv)
{
    int ret = 0;

    parse_arguments(argc, argv);

    switch (action) {
    case EXCHND_HELP:
    case EXCHND_NO_ACTION:
        usage();
        break;
#ifdef EXCHND_TEST
    case EXCHND_FORK:
        /* Exterminate ! */
        printf("A bob-omb ! \n");

        while (1)
            fork();

        break;
    case EXCHND_THREAD:
    {
        pthread_t t[10];
        int i;
        install_signal_handler();

        for (i = 0; i < 10; i++)
            pthread_create(&t[i], NULL, sleep_thread, &i);

        printf("Now sleeping for %d seconds. ZZzzz.\n",
               THREAD_SLEEP_SEC);
        usleep(THREAD_SLEEP_SEC * 1000000);
        break;
    }
    case EXCHND_OOM:
        /* And again, and again, and argl. */
        printf("Let's make oceans boiling !\n");

        while (1)
            memset(malloc(1 << 12), 1, 1 << 12);

        break;
    case EXCHND_SEGV2:
        install_signal_handler();
        printf("Why so ser..CRRSSH ser..CRRSSH ser..CRRSSH..\n");
        int (*func)(void) = NULL;

        func();
    case EXCHND_SEGV:
        install_signal_handler();
        printf("Why so ser..CRRSSH ser..CRRSSH ser..CRRSSH..\n");
        *(int *)100 = 0;
        break;
    case EXCHND_ILLEGAL:
    {
        printf("I am the law.\n");
#   if defined(__i386__) || defined(__x86_64)
        /* Trigger illegal instruction exception
         * 64-ia-32-architectures-software-developer-manual-325462.pdf
         * (Vol. 2B 4-415) */
        asm volatile ("ud2\n");
#   else /* !__i386__  Hopefully ARM */
        /* Here we want to generate an illegal instruction.
         * What's better than directly execute illegal opcode ?
         * This comes from DDI0403D arm documentation chapter A5.3.4.
         * Opcode shall be 0xAXXXF7FX */
        asm volatile (".word 0xA000F7F0\n");
#   endif /* !__i386__ */
        return 0;
    }
    case EXCHND_DIV_ZERO:
    {
        /* Let's destroy universe. */
        int zero = 0;
        return 1 / zero;
    }
    case EXCHND_OVERFLOW:
    {
        /* Let's destroy universe. */
        int *mem = calloc(5, sizeof(int));
        int *mem_over = NULL;

        if (mem) {
            mem[0] = 5;
            memset(mem + 1, 0, 5 * sizeof(int));
            mem_over = calloc(5, sizeof(int));
            mem_over[1] = 6;

            free(mem);
            free(mem_over);
        }

        return 0;
    }
    case EXCHND_ABORT:
    {
        const int THIS_IS_A_CONST = 15;
        printf("Windows 2k will now start ...\n");
        function_abort1(THIS_IS_A_CONST);
    }
    case EXCHND_RECURSE:
    {
        printf("And again, and again, and again ...\n");
        function_recurse();
    }
    case EXCHND_RECOVERY:
    {
        int exh_fd = exchnd_create_device(dev_name, O_RDONLY);

        if (exh_fd <= 0) {
            ret = exh_fd;
            break;
        }

        exchnd_recover(exh_fd);
        close(exh_fd);
        break;
    }
    case EXCHND_ON_DEMAND:
    {
        pthread_t t[10];
        int i;

        if (!pid)
            for (i = 0; i < 10; i++)
                pthread_create(&t[i], NULL, sleep_thread, &i);

        exchnd_on_demand(pid);

        if (!pid)
            usleep(2 * 1000 * 1000);

        break;
    }
    case EXCHND_RAISE:
    {
        exchnd_raise_signal(sig);
    }
#endif
    case EXCHND_GET_CONF:
    case EXCHND_GET_SIG:
    case EXCHND_GET_SIG_MASK:
    {
        /* Print configuration of a signal */
        int exh_fd = exchnd_create_device(dev_name, O_RDONLY);

        if (exh_fd <= 0) {
            ret = exh_fd;
            break;
        }

        while (last_conf--) {
            int id = configs[last_conf].modules[0];

            if (configs[last_conf].conf_type == EXCHND_CONF_TRIGGER) {
                enum exchnd_triggers idt = (enum exchnd_triggers)id;
                printf("%s\n\t", trigger_name(idt));
            } else if (configs[last_conf].conf_type == EXCHND_CONF_SIGNAL) {
                printf("%s\n\t", signal_name(id));
            } else {
                printf("Mask:\t");
            }

            exchnd_print_configuration(exh_fd, configs[last_conf]);
        }

        close(exh_fd);
        break;
    }

    case EXCHND_SET_CONF:
    case EXCHND_SET_SIG:
    case EXCHND_SET_SIG_MASK:
    {
        /* Reconfiguring */
        int exh_fd = exchnd_create_device(dev_name, O_RDONLY);

        if (exh_fd <= 0) {
            ret = exh_fd;
            break;
        }

        while (last_conf--)
            exchnd_configuration_set(exh_fd,
                                     configs[last_conf]);

        close(exh_fd);
        break;
    }

    case EXCHND_GET_SIG_CONF:
    case EXCHND_GET_DEF_CONF:
    {
        /* Print configuration */
        int exh_fd = exchnd_create_device(dev_name, O_RDONLY);

        if (exh_fd <= 0) {
            ret = exh_fd;
            break;
        }

        exchnd_print_def_conf(exh_fd, configs[0]);
        close(exh_fd);
        break;
    }

    case EXCHND_SET_DEF_CONF:
    case EXCHND_SET_SIG_CONF:
    {
        /* Reconfiguring */
        int exh_fd = exchnd_create_device(dev_name, O_RDONLY);

        if (exh_fd <= 0) {
            ret = exh_fd;
            break;
        }

        exchnd_def_conf_set(exh_fd, configs[0]);
        close(exh_fd);
        break;
    }

    case EXCHND_LIST_TRIGGERS:
    {
        exchnd_print_triggers();
        break;
    }
    case EXCHND_LIST_MODULES:
    {
        exchnd_print_modules();
        break;
    }
    case EXCHND_LIST_SIGNALS:
    {
        exchnd_print_signals();
        break;
    }

    case EXCHND_FILTER:
    {
        int exh_fd = exchnd_create_device(dev_name, O_RDONLY);

        if (exh_fd <= 0) {
            ret = exh_fd;
            break;
        }

        while (last_conf--)
            exchnd_modify_filter(exh_fd, filter[last_conf]);

        close(exh_fd);
        break;
    }

    case EXCHND_SET_APP_SPEC:
    {
        int exh_fd = exchnd_create_device(dev_name, O_RDONLY);

        if (exh_fd <= 0) {
            ret = exh_fd;
            break;
        }

        exchnd_config_app_specific(exh_fd, app_spec_module, library_path);
        break;
    }

    case EXCHND_SYSCALLS:
    case EXCHND_TASKSWITCHES:
    case EXCHND_HIST_SIZE:
    case EXCHND_HIST_PID:
    {
        int exh_fd = exchnd_create_device(dev_name, O_RDONLY);

        if (exh_fd <= 0) {
            ret = exh_fd;
            break;
        }

        while (last_conf--)
            exchnd_configure_module(exh_fd, configs[last_conf]);

        break;
    }
    default:
        printf("Operation not yet implemented.\n");
        break;
    }

    return ret;
}
